document.addEventListener("DOMContentLoaded", function() {
    
    const modeSwitch = document.getElementById('switch-sleep-awake');
    
    modeSwitch.addEventListener('change', function() {
        if (this.checked) {
            window.location.href = 'https://miora.ralaivaoalandreo.mmi-velizy.fr/Datavisualisation/artistique.html'; 
            
        } else {
            window.location.href = 'https://miora.ralaivaoalandreo.mmi-velizy.fr/Datavisualisation/carte.html';
        }
    });

    var listeIncendies = [];
    var chartAnneesInstance = null;
    var chartDetailsInstance = null;
    var map = null;
    var geoJsonLayer = null; 
    var layerSelectionne = null;
    var currentCityName = null;
    var chartTendanceInstance = null;
    var chartCorrelationInstance = null;

    // Initialisation du groupe Canicule
    window.layerCanicule = L.layerGroup(); 
    
    


    // =========================================================
    // 1. SYSTÈME D'AIDE
    // =========================================================
    const modal = document.getElementById('explanation-modal');
    const explanationText = document.getElementById('explanation-text');
    const modalTitle = document.getElementById('modal-title');
    const closeModal = document.querySelector('.close-modal');

    window.ouvrirExplication = function(element) {
        if(window.event) { 
            window.event.stopPropagation(); 
            window.event.preventDefault(); 
        }

        const texte = element.getAttribute('data-explanation');
        const titre = element.getAttribute('data-title') || "Information";

        if(explanationText) explanationText.innerHTML = texte;
        if(modalTitle) modalTitle.textContent = titre;
        if(modal) modal.style.display = 'block';
    };

    if(closeModal) closeModal.addEventListener('click', () => { modal.style.display = 'none'; });
    window.addEventListener('click', (e) => { if (e.target === modal) modal.style.display = 'none'; });
    document.addEventListener('keydown', (e) => { if (e.key === 'Escape') modal.style.display = 'none'; });


    // =========================================================
    // 2. LOGIQUE PRINCIPALE
    // =========================================================

    // --- GESTION ACCORDÉON (Fermeture auto) ---
    function initAccordion() {
        const headers = document.querySelectorAll('.accordion-header');
        const allItems = document.querySelectorAll('.accordion-item'); 

        headers.forEach(header => {
            header.addEventListener('click', (e) => {
                if(e.target.closest('button') || e.target.classList.contains('aide-data')) return;

                const item = header.parentElement;
                const estDejaOuvert = item.classList.contains('ouvert');

                // Fermer tous les autres
                allItems.forEach(el => el.classList.remove('ouvert'));

                // Ouvrir si c'était fermé
                if (!estDejaOuvert) {
                    item.classList.add('ouvert');
                }
            });
        });
    }
    initAccordion();

    // --- INITIALISATION DES FILTRES (DOUBLE JEU DE FILTRES) ---
    function initFiltersFromData() {
        // Jeu de filtres 1 (Graphique Historique)
        const selectStart1 = document.getElementById('filter-start');
        const selectEnd1 = document.getElementById('filter-end');
        
        // Jeu de filtres 2 (Graphique Causes)
        const selectStart2 = document.getElementById('filter-start-2');
        const selectEnd2 = document.getElementById('filter-end-2');

        if(!selectStart1 || !selectEnd1 || listeIncendies.length === 0) return;

        const annees = listeIncendies.map(l => parseInt(l["Année"])).filter(a => !isNaN(a));
        if (annees.length === 0) return;

        const minDataYear = Math.min(...annees);
        const maxDataYear = Math.max(...annees);

        // Fonction pour remplir un select
        const remplirSelect = (sel) => {
            if(!sel) return;
            sel.innerHTML = "";
            for(let y = minDataYear; y <= maxDataYear; y++) {
                sel.add(new Option(y, y));
            }
        };

        // On remplit les 4 selects
        remplirSelect(selectStart1);
        remplirSelect(selectEnd1);
        remplirSelect(selectStart2);
        remplirSelect(selectEnd2);

        // Valeurs par défaut
        if(selectStart1) selectStart1.value = minDataYear;
        if(selectEnd1) selectEnd1.value = maxDataYear;
        if(selectStart2) selectStart2.value = minDataYear;
        if(selectEnd2) selectEnd2.value = maxDataYear;

        const pPeriode = document.getElementById('periode-ville');
        if(pPeriode) pPeriode.textContent = `Données historiques ${minDataYear}-${maxDataYear}`;

        // --- ÉCOUTEURS D'ÉVÉNEMENTS INDÉPENDANTS ---

        // Mise à jour Graphique 1 (Surface & Stats)
        function updateGraph1() {
            if(currentCityName) updateGraphique1UI(currentCityName);
        }

        if(selectStart1 && selectEnd1) {
            selectStart1.addEventListener('change', function() {
                if(parseInt(selectStart1.value) > parseInt(selectEnd1.value)) selectEnd1.value = selectStart1.value;
                updateGraph1();
            });
            selectEnd1.addEventListener('change', function() {
                if(parseInt(selectEnd1.value) < parseInt(selectStart1.value)) selectStart1.value = selectEnd1.value;
                updateGraph1();
            });
        }

        // Mise à jour Graphique 2 (Causes)
        function updateGraph2() {
            if(currentCityName) updateGraphique2UI(currentCityName);
        }

        if(selectStart2 && selectEnd2) {
            selectStart2.addEventListener('change', function() {
                if(parseInt(selectStart2.value) > parseInt(selectEnd2.value)) selectEnd2.value = selectStart2.value;
                updateGraph2();
            });
            selectEnd2.addEventListener('change', function() {
                if(parseInt(selectEnd2.value) < parseInt(selectStart2.value)) selectStart2.value = selectEnd2.value;
                updateGraph2();
            });
        }
    }

    // --- GESTION DES ONGLETS ---
    const btnIncendie = document.getElementById('btn-incendie');
    const btnCanicule = document.getElementById('btn-canicule');
    const wrapperIncendie = document.getElementById('mode-incendie-wrapper');
    const wrapperCanicule = document.getElementById('mode-canicule-wrapper');

    if(btnIncendie) btnIncendie.addEventListener('click', () => activerMode('incendie'));
    if(btnCanicule) btnCanicule.addEventListener('click', () => activerMode('canicule'));

    function activerMode(mode) {
        const legendeIncendie = document.getElementById('legende-incendie');
        const legendeCanicule = document.getElementById('legende-canicule');
    
        if (mode === 'incendie') {
            btnIncendie.classList.add('actif');
            btnCanicule.classList.remove('actif');
            wrapperIncendie.classList.remove('cache');
            wrapperIncendie.classList.add('actif');
            wrapperCanicule.classList.add('cache');
            
            if (geoJsonLayer) geoJsonLayer.addTo(window.map);
            if (window.layerCanicule) window.layerCanicule.removeFrom(window.map);
    
            if(legendeIncendie) legendeIncendie.classList.remove('cache');
            if(legendeCanicule) legendeCanicule.classList.add('cache');
            
            setTimeout(() => creerGraphiqueTendanceTemporelle(), 100);
    
        } else {
            btnCanicule.classList.add('actif');
            btnIncendie.classList.remove('actif');
            wrapperCanicule.classList.remove('cache');
            wrapperCanicule.classList.add('actif');
            wrapperIncendie.classList.add('cache');
    
            if (geoJsonLayer) geoJsonLayer.removeFrom(window.map);
            if (window.layerCanicule) window.layerCanicule.addTo(window.map);
    
            if(legendeIncendie) legendeIncendie.classList.add('cache');
            if(legendeCanicule) legendeCanicule.classList.remove('cache');
    
            setTimeout(() => creerGraphiqueCorrelationEcartIncendies(), 100);
        }
    }

    // --- CARTE LEAFLET ---
    window.map = L.map('map').setView([44.8378, -0.5795], 11);
    L.tileLayer('https://{s}.basemaps.cartocdn.com/dark_all/{z}/{x}/{y}{r}.png', {
        attribution: '&copy; OpenStreetMap &copy; CARTO',
        maxZoom: 19
    }).addTo(window.map);

    // --- CHARGEMENT DONNÉES ---
    fetch('incendie.json?v=' + Date.now()) 
        .then(r => r.json())
        .then(data => {
            listeIncendies = data;
            initFiltersFromData();
            chargerGeoJSON();
            
            // AJOUT : Initialiser les graphiques statistiques
            initialiserGraphiquesStatistiques();
        });

    function chargerGeoJSON() {
        fetch('commune.geojson')
        .then(r => r.json())
        .then(geoData => {
            geoJsonLayer = L.geoJSON(geoData, {
                style: function(feature) {
                    var nom = feature.properties.nom_com || feature.properties.nom;
                    
                    var surface = calculerSurfaceTotaleVille(nom);
                    return {
                        fillColor: choisirCouleur(surface),
                        weight: 1, opacity: 1, color: '#333', 
                        fillOpacity: surface > 0 ? 0.7 : 0.2 
                    };
                },
                onEachFeature: function(feature, layer) {
                    var nom = feature.properties.nom_com || feature.properties.nom;
                    
                    layer.bindTooltip(nom, {
                        permanent: false,      // false = seulement au survol
                        direction: 'center',   // Apparait au centre de la commune
                        className: 'label-commune', // Le style CSS qu'on a créé
                        opacity: 1
                    });
                    
                    layer.on('mouseover', function() {
                        if (document.getElementById('btn-canicule').classList.contains('actif')) return;
                        if (layer !== layerSelectionne) {
                            this.setStyle({ weight: 3, color: '#fff', fillOpacity: 0.9 });
                            this.bringToFront();
                        }
                    });

                    layer.on('mouseout', function() {
                        if (layer !== layerSelectionne) geoJsonLayer.resetStyle(this);
                    });

                    layer.on('click', function(e) { 
                        if (document.getElementById('btn-canicule').classList.contains('actif')) return;
                        if (layerSelectionne) geoJsonLayer.resetStyle(layerSelectionne);
                        layerSelectionne = layer;
                        layer.setStyle({ weight: 4, color: '#ff4500', fillOpacity: 1 });
                        window.map.flyTo(e.latlng, 12, { duration: 1.5 });
                        
                        currentCityName = nom;
                        
                        // On lance les deux mises à jour
                        updateGraphique1UI(nom);
                        updateGraphique2UI(nom);
                    });
                }
            }).addTo(window.map);
        });
    }

    // --- OUTILS ---
    function nettoyerNom(t) {
        if (!t) return "";
        return String(t).toUpperCase().normalize("NFD").replace(/[\u0300-\u036f]/g, "").replace(/-/g, " ").replace(/^ST /, "SAINT ").trim();
    }
    function lireNombre(v) { return v ? Number(String(v).replace(',', '.')) : 0; }

    function calculerSurfaceTotaleVille(nomVille) {
        let total = 0;
        const nomPropre = nettoyerNom(nomVille);
        listeIncendies.filter(l => nettoyerNom(l["Nom de la commune"]) === nomPropre)
                      .forEach(l => total += lireNombre(l["Surface parcourue (m2)"]));
        return Math.round(total / 10000 * 100) / 100;
    }

    function choisirCouleur(s) {
        if (s > 100) return '#4a0000'; if (s > 50) return '#8b0000';
        if (s > 10) return '#ff0000'; if (s > 1) return '#ff4500';
        if (s > 0) return '#ffa500'; return '#2d5a3d';
    }

    // --- MISE A JOUR GRAPHIQUE 1 (Surface & Stats) ---
    function updateGraphique1UI(nomVille) {
        var titre = document.getElementById('nom-ville');
        if(titre) titre.textContent = nomVille;

        // Récupération Filtres 1
        const startYear = parseInt(document.getElementById('filter-start').value);
        const endYear = parseInt(document.getElementById('filter-end').value);
        const nomPropre = nettoyerNom(nomVille);
        
        const feuxVille = listeIncendies.filter(l => {
            const annee = parseInt(l["Année"]);
            return nettoyerNom(l["Nom de la commune"]) === nomPropre &&
                   annee >= startYear && annee <= endYear;
        });

        let totalM2 = 0;
        let anneeMax = "N/A";
        let maxSurface = 0;
        const surfacesParAnnee = {}; 

        for (let y = startYear; y <= endYear; y++) {
            surfacesParAnnee[y] = 0;
        }

        feuxVille.forEach(feu => {
            const surf = lireNombre(feu["Surface parcourue (m2)"]);
            const annee = parseInt(feu["Année"]);
            
            totalM2 += surf;
            if (surfacesParAnnee[annee] !== undefined) surfacesParAnnee[annee] += surf;
            if (surf > maxSurface) { maxSurface = surf; anneeMax = annee; }
        });

        // ANIMATION COMPTEURS (Liés au filtre 1)
        function animateValue(idElement, start, end, duration, decimales, suffixe) {
            const obj = document.getElementById(idElement);
            if (!obj) return;
            let startTimestamp = null;
            const step = (timestamp) => {
                if (!startTimestamp) startTimestamp = timestamp;
                const progress = Math.min((timestamp - startTimestamp) / duration, 1);
                const easeProgress = 1 - Math.pow(1 - progress, 3);
                const current = start + (easeProgress * (end - start));
                obj.textContent = current.toFixed(decimales) + (suffixe || '');
                if (progress < 1) { window.requestAnimationFrame(step); } 
                else { obj.textContent = end.toFixed(decimales) + (suffixe || ''); }
            };
            window.requestAnimationFrame(step);
        }

        animateValue('stat-surface', 0, (totalM2 / 10000), 1000, 2, ' ha');
        animateValue('stat-nb', 0, feuxVille.length, 1000, 0, '');
        
        var elAnnee = document.getElementById('stat-annee-max');
        if(elAnnee) elAnnee.textContent = anneeMax !== "N/A" ? anneeMax : "-";

        // GRAPH 1 (Animation Progressive)
        var canvasAnnees = document.getElementById('graphiqueAnnees');
        if(canvasAnnees) {
            const ctx = canvasAnnees.getContext('2d');
            if (chartAnneesInstance) chartAnneesInstance.destroy();

            const labelsAnnees = Object.keys(surfacesParAnnee);
            const dataAnnees = labelsAnnees.map(y => (surfacesParAnnee[y] / 10000).toFixed(2));

            const totalDuration = 2000;
            const delayBetweenPoints = totalDuration / dataAnnees.length;
            const previousY = (ctx) => ctx.index === 0 ? ctx.chart.scales.y.getPixelForValue(0) : ctx.chart.getDatasetMeta(ctx.datasetIndex).data[ctx.index - 1].getProps(['y'], true).y;
            
            const progressiveAnimation = {
                x: {
                    type: 'number',
                    easing: 'linear',
                    duration: delayBetweenPoints,
                    from: NaN, 
                    delay(ctx) {
                        if (ctx.type !== 'data' || ctx.xStarted) return 0;
                        ctx.xStarted = true;
                        return ctx.index * delayBetweenPoints;
                    }
                },
                y: {
                    type: 'number',
                    easing: 'linear',
                    duration: delayBetweenPoints,
                    from: previousY,
                    delay(ctx) {
                        if (ctx.type !== 'data' || ctx.yStarted) return 0;
                        ctx.yStarted = true;
                        return ctx.index * delayBetweenPoints;
                    }
                }
            };

            chartAnneesInstance = new Chart(ctx, {
                type: 'line',
                data: {
                    labels: labelsAnnees,
                    datasets: [{
                        label: 'Surface brûlée (ha)',
                        data: dataAnnees,
                        borderColor: '#ff4500',
                        backgroundColor: 'rgba(255, 69, 0, 0.2)',
                        borderWidth: 2,
                        fill: true,
                        pointRadius: 3,
                        pointHoverRadius: 6
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    animation: progressiveAnimation,
                    interaction: { intersect: false, mode: 'index' },
                    scales: {
                        x: { grid: { display: false }, ticks: { color: '#888' } },
                        y: { beginAtZero: true, grid: { color: '#444' }, ticks: { color: '#888' } }
                    },
                    plugins: { legend: { display: false } }
                }
            });
        }
    }

    // --- MISE A JOUR GRAPHIQUE 2 (Causes) ---
    function updateGraphique2UI(nomVille) {
        
        // Récupération Filtres 2 (Nouveaux IDs)
        const s2 = document.getElementById('filter-start-2');
        const e2 = document.getElementById('filter-end-2');
        
        // Sécurité si les filtres 2 n'existent pas encore
        if(!s2 || !e2) return;

        const startYear = parseInt(s2.value);
        const endYear = parseInt(e2.value);
        const nomPropre = nettoyerNom(nomVille);
        
        const feuxVille = listeIncendies.filter(l => {
            const annee = parseInt(l["Année"]);
            return nettoyerNom(l["Nom de la commune"]) === nomPropre &&
                   annee >= startYear && annee <= endYear;
        });

        const historiqueCauses = {};
        for (let y = startYear; y <= endYear; y++) {
            historiqueCauses[y] = { 'Inconnue': 0, 'Accidentelle': 0, 'Malveillance': 0, 'Naturelle': 0 };
        }

        feuxVille.forEach(feu => {
            const annee = parseInt(feu["Année"]);
            let natureRaw = (feu["Nature"] || "Inconnue").toLowerCase();
            let category = 'Inconnue';

            if (natureRaw.includes('foudre') || natureRaw.includes('naturelle')) category = 'Naturelle';
            else if (natureRaw.includes('malveillance') || natureRaw.includes('incendiaire') || natureRaw.includes('criminel')) category = 'Malveillance';
            else if (natureRaw.includes('travaux') || natureRaw.includes('loisirs') || natureRaw.includes('imprudence') || natureRaw.includes('accidentelle') || natureRaw.includes('involontaire')) category = 'Accidentelle';

            if (historiqueCauses[annee]) {
                historiqueCauses[annee][category]++;
            }
        });

        // GRAPH 2 (Mixte Animation)
        var canvasDetails = document.getElementById('graphiqueDetails');
        if(canvasDetails) {
            const ctxDetails = canvasDetails.getContext('2d');
            if (chartDetailsInstance) chartDetailsInstance.destroy();

            const labelsHist = Object.keys(historiqueCauses);
            const dataMal = labelsHist.map(y => historiqueCauses[y]['Malveillance']);
            const dataAcc = labelsHist.map(y => historiqueCauses[y]['Accidentelle']);
            const dataNat = labelsHist.map(y => historiqueCauses[y]['Naturelle']);
            const dataInc = labelsHist.map(y => historiqueCauses[y]['Inconnue']);
            const dataTotal = labelsHist.map(y => 
                historiqueCauses[y]['Malveillance'] + 
                historiqueCauses[y]['Accidentelle'] + 
                historiqueCauses[y]['Naturelle'] + 
                historiqueCauses[y]['Inconnue']
            );

            // Anim ligne
            const totalDurationLine = 2000;
            const delayBetweenPointsLine = totalDurationLine / dataTotal.length;
            const previousYLine = (ctx) => ctx.index === 0 ? ctx.chart.scales.y.getPixelForValue(0) : ctx.chart.getDatasetMeta(ctx.datasetIndex).data[ctx.index - 1].getProps(['y'], true).y;
            
            const progressiveAnimationLine = {
                x: {
                    type: 'number',
                    easing: 'linear',
                    duration: delayBetweenPointsLine,
                    from: NaN, 
                    delay(ctx) {
                        if (ctx.type !== 'data' || ctx.xStarted) return 0;
                        ctx.xStarted = true;
                        return ctx.index * delayBetweenPointsLine;
                    }
                },
                y: {
                    type: 'number',
                    easing: 'linear',
                    duration: delayBetweenPointsLine,
                    from: previousYLine,
                    delay(ctx) {
                        if (ctx.type !== 'data' || ctx.yStarted) return 0;
                        ctx.yStarted = true;
                        return ctx.index * delayBetweenPointsLine;
                    }
                }
            };

            // Anim barres (delayed)
            let delayed;
            const barDelayAnimation = {
                onComplete: () => { delayed = true; },
                delay: (context) => {
                    let delay = 0;
                    if (context.type === 'data' && context.mode === 'default' && !delayed) {
                        delay = context.dataIndex * 300 + context.datasetIndex * 100;
                    }
                    return delay;
                }
            };

            chartDetailsInstance = new Chart(ctxDetails, {
                type: 'bar',
                data: {
                    labels: labelsHist,
                    datasets: [
                        { 
                            type: 'line', 
                            label: 'Total', 
                            data: dataTotal, 
                            borderColor: '#fff', 
                            borderWidth: 2,
                            animation: progressiveAnimationLine 
                        },
                        { label: 'Malveillance', data: dataMal, backgroundColor: '#d63031', stack: 'Stack 0' },
                        { label: 'Accident', data: dataAcc, backgroundColor: '#e67e22', stack: 'Stack 0' },
                        { label: 'Naturelle', data: dataNat, backgroundColor: '#27ae60', stack: 'Stack 0' },
                        { label: 'Inconnue', data: dataInc, backgroundColor: '#636e72', stack: 'Stack 0' }
                    ]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    animation: barDelayAnimation,
                    interaction: { mode: 'index', intersect: false },
                    scales: {
                        x: { stacked: true, grid: { display: false }, ticks: { color: '#888' } },
                        y: { stacked: true, grid: { color: '#444' }, ticks: { color: '#888' } }
                    }
                }
            });
        }
    }
    
    // Fonction globale d'affichage qui déclenche les deux mises à jour
    window.afficherInfosVille = function(nomVille) {
        // Cette fonction sert désormais de routeur
        // Elle met à jour les deux graphiques simultanément lors du clic sur la carte
        updateGraphique1UI(nomVille);
        updateGraphique2UI(nomVille);
    };

});